/* -*-c++-*- Copyright (C) 2018 Advanced Driver Information Technology.
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
*/

#include "WLClientContext.hpp"

#include <string.h>
#include <poll.h>
#include <errno.h>
#include <map>
#include <wayland-client.h>

#include "WLWindowController.hpp"
#include "WLCompositor.hpp"
#include "WLInterface.hpp"
#include "WLIVIShell.hpp"
#include "WLRegistry.hpp"
#include "WLSeat.hpp"
#include "WLShell.hpp"
#ifdef SERVERINFO_ENABLED
#include "WLServerInfo.hpp"
#endif

namespace WaylandBackend
{
    class GlobalEventHandler : public WLRegistry
    {
    public:
        GlobalEventHandler(struct wl_display *wlDisplay,
                           struct wl_event_queue *eventQ);
        virtual ~GlobalEventHandler();
        void registryGlobal(struct wl_registry* registry,
                                    uint32_t name, const char* interface,
                                    uint32_t version);

        void registryGlobalRemove(struct wl_registry* registry,
                                          uint32_t name);

        void notifySurfaceDestroy(struct wl_surface* wlSurface);

        WLCompositor*                     _wlCompositor;
        WLShellInterface*                 _wlActiveShell;

    private:
        struct wl_event_queue*            _wlEventQ;
        std::map<uint32_t, WLInterface *> _wlProtocolObjMap;

    };
}

using namespace WaylandBackend;

GlobalEventHandler::GlobalEventHandler(struct wl_display *wlDisplay,
                   struct wl_event_queue *eventQ):
 WLRegistry(wlDisplay, eventQ)
,_wlCompositor(NULL)
,_wlActiveShell(NULL)
,_wlEventQ(eventQ)
,_wlProtocolObjMap()
{

}

GlobalEventHandler::~GlobalEventHandler()
{
    std::map<uint32_t, WLInterface*>::iterator protocolObj =
            _wlProtocolObjMap.begin();

    for (;protocolObj != _wlProtocolObjMap.end(); protocolObj++)
    {
        delete protocolObj->second;
    }
    _wlProtocolObjMap.clear();
}

void
GlobalEventHandler::registryGlobal(struct wl_registry* registry,
                                   uint32_t name, const char* interface,
                                   uint32_t version)

{
    WLInterface *wlServerIf = NULL;
    WLIVIShell *wlIviShell;
    WLShell *wlShell;

    int ret;
    if (!strcmp(interface, "wl_compositor"))
    {
        _wlCompositor = new WLCompositor(_wlEventQ);
        wlServerIf = _wlCompositor;
    }
    if (!strcmp(interface, "wl_seat"))
    {
        wlServerIf = new WLSeat(_wlEventQ);
    }

#ifdef SERVERINFO_ENABLED
    WLServerInfo *serverInfo;
    if (!strcmp(interface, "serverinfo"))
    {
        serverInfo = new WLServerInfo(_wlEventQ);
        wlServerIf = serverInfo;
        _wlActiveShell = serverInfo;
    }
#endif

    if (!strcmp(interface, "ivi_application"))
    {
        wlIviShell = new WLIVIShell(_wlEventQ);
        wlServerIf = wlIviShell;
        _wlActiveShell = wlIviShell;
    }
    if (!strcmp(interface, "wl_shell"))
    {
        wlShell = new WLShell(_wlEventQ);
        wlServerIf = wlShell;
        _wlActiveShell = wlShell;
    }

    if (NULL != wlServerIf)
    {
        ret = wlServerIf->init(registry, name, interface, version);

        if (ret >= 0)
            _wlProtocolObjMap.insert(std::pair<uint32_t, WLInterface*>
                                     (name, wlServerIf));
        else
        {
            //TODO: LOG_ERROR("init failed for interface" << interface);
        }
    }
    else
    {
        //TODO: LOG_ERROR("memory allocation failed for interface" << interface);
    }
}

void
GlobalEventHandler::registryGlobalRemove(struct wl_registry* registry,
                                         uint32_t name)
{
    std::map<uint32_t, WLInterface*>::iterator found;
    WLInterface *wlServerIf;

    found = _wlProtocolObjMap.find(name);
    if (found != _wlProtocolObjMap.end())
    {
        wlServerIf = found->second;
        _wlProtocolObjMap.erase(name);
        delete wlServerIf;
    }
}

void
GlobalEventHandler::notifySurfaceDestroy(struct wl_surface* wlSurface)
{
    std::map<uint32_t, WLInterface*>::iterator protocolObj;
    WLInterface *wlServerIf;

    for (protocolObj = _wlProtocolObjMap.begin();
         protocolObj != _wlProtocolObjMap.end(); protocolObj++)
    {
        wlServerIf = protocolObj->second;
        wlServerIf->surfaceDestroyNotification(wlSurface);
    }
}

WLClientContext::WLClientContext(const char * connectionName)
{
    pthread_mutex_init(&_dispatchProtectMutex, NULL);

    _wlDisplay = wl_display_connect(connectionName);
    if (NULL != _wlDisplay)
    {
        _wlEventQ = wl_display_create_queue(_wlDisplay);

        _wlEventHandler = new GlobalEventHandler(_wlDisplay, _wlEventQ);

        if (NULL != _wlEventHandler)
        {
            wl_display_roundtrip_queue(_wlDisplay, _wlEventQ);
            wl_display_roundtrip_queue(_wlDisplay, _wlEventQ);
        }
    }
}

void*
WLClientContext::getDisplay()
{
    return (void*)_wlDisplay;
}

WLWindowController*
WLClientContext::createWindowController(WLSurfaceEvent *wlInputHandler)
{
    struct wl_surface *wlSurface;
    WLWindowController *winCtrl = NULL;
    wlSurface = _wlEventHandler->_wlCompositor->createSurface(wlInputHandler);
    if (NULL != wlSurface)
    {
        if (NULL !=_wlEventHandler->_wlActiveShell)
        {
            winCtrl = new WLWindowController(wlSurface, this);
        }
        else
        {
            _wlEventHandler->notifySurfaceDestroy(wlSurface);
            _wlEventHandler->_wlCompositor->destroySurface(wlSurface);
        }
    }
    return winCtrl;
}

int
WLClientContext::destroyWindowController(WLWindowController *wlWinCtrl)
{
    int ret = -EINVAL;
    struct wl_surface *wlSurface;

    if (NULL != wlWinCtrl)
    {
        wlSurface = (struct wl_surface *)wlWinCtrl->getWlSurface();

        delete wlWinCtrl;
        pthread_mutex_lock(&_dispatchProtectMutex);
        _wlEventHandler->notifySurfaceDestroy(wlSurface);
        _wlEventHandler->_wlCompositor->destroySurface(wlSurface);
        pthread_mutex_unlock(&_dispatchProtectMutex);
        ret = 0;
    }
    return ret;
}


void*
WLClientContext::getWlOuput(const char *outputName)
{
    /*TODO:To be Implemented*/
    return NULL;
}

void*
WLClientContext::getActiveShell()
{
    return (void*)_wlEventHandler->_wlActiveShell;
}


int
WLClientContext::pollForEvents(short int events, int timeout)
{
    struct pollfd pfd[1];
    int ret = 0;

    pfd[0].fd = wl_display_get_fd(_wlDisplay);
    pfd[0].events = events;

    do
    {
        ret = poll(pfd, 1, timeout);
    } while ((ret == -1) && (errno == EINTR));

    return ret;
}

int
WLClientContext::flushWLDisplay()
{
    int ret;
    bool flushDisplay = true;

    while (flushDisplay)
    {
        flushDisplay = false;
        ret = wl_display_flush(_wlDisplay);

        if ((ret == -1) && (errno == EAGAIN))
        {
            ret = pollForEvents(POLLOUT, -1);
            if (ret >= 0)
                flushDisplay = true;
        }
    }

    return ret;
}

int
WLClientContext::dispatchEvents(int timeout)
{
    int ret = 0;
    bool eventsPending = false;

    if ((NULL != _wlDisplay) && (NULL != _wlEventQ))
    {
        //Flush already existing events in the queue
        pthread_mutex_lock(&_dispatchProtectMutex);
        if (-1 == wl_display_prepare_read_queue(_wlDisplay, _wlEventQ))
        {
            ret = wl_display_dispatch_queue_pending(_wlDisplay, _wlEventQ);
            eventsPending = true;
        }
        pthread_mutex_unlock(&_dispatchProtectMutex);

        if ((ret >= 0) && !eventsPending)
        {
            ret = flushWLDisplay();
            if ((ret >= 0) || ((ret < 0) && (errno == EPIPE)))
            {
                //Continue reading in case of EPIPE as it helps
                //to read any protocol errors and aids further debugging
                ret = pollForEvents(POLLIN, timeout);
                if (ret > 0)
                {
                    ret = wl_display_read_events(_wlDisplay);
                    if (ret >= 0)
                    {
                        pthread_mutex_lock(&_dispatchProtectMutex);
                        ret = wl_display_dispatch_queue_pending(_wlDisplay,
                                                                _wlEventQ);
                        pthread_mutex_unlock(&_dispatchProtectMutex);
                    }
                }
                else
                    wl_display_cancel_read(_wlDisplay);
            }
            else
                wl_display_cancel_read(_wlDisplay);
        }
    }
    return ret;
}

WLClientContext::~WLClientContext()
{
    if (NULL != _wlEventHandler)
        delete _wlEventHandler;

    if (NULL != _wlEventQ)
        wl_event_queue_destroy(_wlEventQ);

    if (NULL != _wlDisplay)
        wl_display_disconnect(_wlDisplay);

    pthread_mutex_destroy(&_dispatchProtectMutex);
}
